(************************************************************************
 *                                                                      *
 *   Ant Movie Catalog 4.x                                              *
 *   (C) 2000-2011 Antoine Potten, Mickal Vanneufville                 *
 *   http://www.antp.be/software                                        *
 *                                                                      *
 ************************************************************************
 *                                                                      *
 *   This program is free software; you can redistribute it and/or      *
 *   modify it under the terms of the GNU General Public License        *
 *   as published by the Free Software Foundation; either version 2     *
 *   of the License, or (at your option) any later version.             *
 *                                                                      *
 *   This program is distributed in the hope that it will be useful,    *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of     *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 *   GNU General Public License for more details.                       *
 *                                                                      *
 ************************************************************************)

unit ProgramSettings;

interface

uses
  Classes, Controls, IniFiles,

  JvSimpleXml,

  ConstValues, MovieClass, functions_files;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

type

TComboOptions = record
  Contents:           TStrings;
  AutoAdd:             Boolean;
  Sort:                Boolean;
  AutoComplete:        Boolean;
  UseCatalogValues:    Boolean;
end;

TFolder = class(TObject)
  private
    Path:       string;
    PathType:   Integer;
    procedure   UpdateFolder(const newPath: string);
    function    ReturnPath: string;
  public
    constructor Create;
    destructor  Destroy; override;
    property    FolderType: integer read PathType;
    property    Value: string read ReturnPath write UpdateFolder;
    procedure   SetFolder(const newPath: string; const newType: Integer);
    procedure   Load(Root: TJvSimpleXmlElem; nr: Integer); overload;
    procedure   Load(ini: TMemIniFile; nr: Integer); overload;
    procedure   Save(Items: TJvSimpleXmlElems; const nr: Integer);
end;

TSettings = class(TObject)
  private
    FXml:             TJvSimpleXml;
    FLanguage:        string;
  public
    version:            string;
    rMain:              record
      WindowState:        Integer;
      WindowWidth:        Integer;
      WindowHeight:       Integer;
      WindowLeft:         Integer;
      WindowTop:          Integer;
      Statusbar:          Boolean;
      ListWidth:          Integer;
      GridMode:           Boolean;
      ListAlClient:       Boolean;
      DisplayHTML:        Boolean;
      DisplayThumbnails:  Boolean;
      PictureDockedHeight:Integer;
      Group:              Integer;
      MRU:                TStringList;
      FindWholefield:     Boolean;
      FindField:          Integer;
      SortField:          Integer;
      ColumnSettings:     string;
      Toolbars:           TJvSimpleXmlElem;
      PictureWinLeft:     Integer;
      PictureWinTop:      Integer;
    end;
    rExport:            record
      WindowState:        Integer;
      WindowWidth:        Integer;
      WindowHeight:       Integer;
      OrderBy:            Integer;
      OrderFields:        TStringList;
      Includemov:         Integer;
      CSVDelimiter:       string;
      CSVBloc:            string;
      CSVLinebreaks:      string;
      CSVColumnTitles:    Boolean;
      CSVFields:          TStringList;
      SQLUpdate:          Boolean;
      SQLDrop:            Boolean;
      SQLCreate:          Boolean;
      SQLFields:          TStringList;
      SQLTableName:       string;
      SQLLinebreaks:      string;
      HTMLLastTemplate:   string;
      HTMLLastTemplate2:  string;
      HTMLMRU:            TStringList;
      ImgFileName:        Integer;
      ExportBoth:         Boolean;
    end;
    rImport:            record
      WindowState:        Integer;
      WindowWidth:        Integer;
      WindowHeight:       Integer;
      AllowDup:           Boolean;
      AutoAssign:         Boolean;
      PicImport:          Integer;
      rCsv:               record
        Delim:              string;
        Quote:              string;
        FirstLineHeaders:   Boolean;
        Linebreaks:         string;
        Fields:             TStringList;
      end;
      rQuery:               record
        From:               string;
        Where:              string;
      end;
      LastFiles:          TStringList;
    end;
    rPrint:             record
      WindowState:        Integer;
      WindowWidth:        Integer;
      WindowHeight:       Integer;
      OrderBy:            Integer;
      OrderFields:        TStringList;
      Includemov:         Integer;
      ReportsListWidth:   Integer;
    end;
    rScripts:           record
      WindowState:        Integer;
      WindowWidth:        Integer;
      WindowHeight:       Integer;
      ScriptMRU:          TStringList;
      ScriptRemember1:    string;
      ScriptRemember2:    string;
      Includemov:         Integer;
      LangFilterExcl:     string;
      ListViewStyle:      Integer;
      ShowResGetInfo:     Boolean;
      ShowResScripting:   Boolean;
      CloseWinGetInfo:    Boolean;
      CloseWinScripting:  Boolean;
      AllowClearGetInfo:  Boolean;
      AllowClearScripting:Boolean;
      rProperties:        record
        WindowWidth:        Integer;
        WindowHeight:       Integer;
      end;
      rResults:           record
        WindowWidth:        Integer;
        WindowHeight:       Integer;
      end;
      rPickList:           record
        WindowWidth:        Integer;
        WindowHeight:       Integer;
      end;
      rPickTree:           record
        WindowWidth:        Integer;
        WindowHeight:       Integer;
      end;
      Toolbars:           TJvSimpleXmlElem;
    end;
    rLoan:              record
      WindowState:        Integer;
      WindowWidth:        Integer;
      WindowHeight:       Integer;
      LvMoviesHeight:     Integer;
      LvMoviesColWidth:   Integer;
      LvMoviesSort:       Integer;
      LvNamesWidth:       Integer;
      LvNamesSort:        Integer;
      LvLentSort:         Integer;
      IncludeSameNum:     Boolean;
      IncludeSameLabel:   Boolean;
    end;
    rProperties:        record
      WindowWidth:        Integer;
      WindowHeight:       Integer;
    end;
    rStatistics:        record
      WindowState:        Integer;
      WindowWidth:        Integer;
      WindowHeight:       Integer;
      EmptyMonths:        Boolean;
      Legend:             Boolean;
      Labels:             Boolean;
      Group:              Boolean;
      Includemov:         Integer;
    end;
    rRenumber:          record
      OrderBy:            Integer;
      OrderFields:        TStringList;
    end;
    rOptions:           record
      WindowWidth:        Integer;
      WindowHeight:       Integer;
      rDisplay:           record
        ImagesInMenu:       Boolean;
        OfficeXP:           Boolean;
        IconSet:            string;
        ColorType:          Integer;
        Logo:               Boolean;
        SoftBorders:        Boolean;
      end;
      rFiles:             record
//        XMLHeader:          string;
        Backup:             Boolean;
        RecentFiles:        Integer;
        AutoLoad:           Boolean;
        AutoLoadFile:       string;
        AutoLoadLast:       Boolean;
        AutoLoadCF:         Boolean;
        AutoLoadCFFile:     string;
        History:            Boolean;
        HistoryFile:        string;
        HTMLTemplateFile:   string;
      end;
      rMovieList:         record
        TitleColumn:        Integer;
        UsePrefixes:        Boolean;
        Prefixes:           TStringList;
        ConfirmDelete:      Boolean;
        ConfirmUndo:        Boolean;
        Checkboxes:         Boolean;
        HotTrack:           Boolean;
        EnhancedScrollbars: Boolean;
        MovieNumColumn:     Boolean;
        GridTextSize:       Integer;
        ShortcutPrev:       TShortcut;
        ShortcutNext:       TShortcut;
        GroupCount:         Boolean;
        GroupExpand:        Boolean;
        GroupsAbove:        Boolean;
        GroupUnique:        Boolean;
        GroupMulti:         Boolean;
        GroupMultiSep:      Char;
        GroupMultiRmAllP:   Boolean;
        GroupMultiAddPatch: Boolean;
        SortGroupsByCount:  Boolean;
      end;
      rMovieInformation:  record
        AskNumber:          Boolean;
        FirstAvailable:     Boolean;
        AddScript:          Boolean;
        AddFiles:           Boolean;
        RatingTrunc:        Boolean;
        SetCurrentDate:     Boolean;
        PictureBackground:  Integer;
        PictureFitWindow:   Boolean;
        PictureInfo:        Boolean;
        ImportMediaLabel:   Boolean;
        ImportFileName:     Boolean;
        ImportFileNameURL:  Boolean;
        ImportSize:         Boolean;
        ImportSizeString:   Boolean;
        ImportSizeUnit:     TFileSizeUnit;
        ImportInternalAVI:  Boolean;
        ImportResolution:   Boolean;
        ImportLength:       Boolean;
        ImportFramerate:    Boolean;
        ImportVideoCodec:   Boolean;
        ImportVideoBitrate: Boolean;
        ImportAudioCodec:   Boolean;
        ImportAudioBitrate: Boolean;
        ImportLanguages:    Boolean;
        ImportSubtitles:    Boolean;
        rPicImport:         record
          GetInfoMethod:      Integer;
          ScriptingMethod:    Integer;
          Naming:             Integer;
          CatalogPrefix:      Boolean;
        end;
        SearchSites:        TStringList;
        ComboSameForAll:    Boolean;
        rCombo:             array[0..9] of TComboOptions;
        rDefaultMovie:      record
          Values:             TMovie;
          WindowWidth:        Integer;
          WindowHeight:       Integer;
        end;
      end;
      rScripting:         record
        ScriptAutorun:     Boolean;
        Proxy:              Boolean;
        ProxyServer:        string;
        ProxyPort:          Integer;
        ProxyUsername:      string;
        ProxyPassword:      string;
        KeepConnection:     Boolean;
        HTTP10:             Boolean;
      end;
      rExport:            record
        LoadTemplate:       Boolean;
        Linebreak:          string;
        ForcePicSizeW:      Integer;
        ForcePicSizeH:      Integer;
        SQLDate:            string;
        RememberLastFile:   Boolean;
        LastFile:           string;
        OpenExportedFile:   Boolean;
        ExpFileStyle:       Integer;
        ExpFileExt:         string;
        AddZeroes:          Boolean;
        CopyPictures:       Boolean;
        CopyPicturesNew:    Boolean;
      end;
      rFolders:           array[0..8] of TFolder;
      SameFolderForAll:   Boolean;
      rLanguage:          record
        Language:           string;
      end;
    end;
    rCustomFieldsManager: record
      WindowWidth:        Integer;
      WindowHeight:       Integer;
    end;
    constructor   Create;
    destructor    Destroy;override;
    procedure     Load;
    procedure     Save;
    property      Language: string read FLanguage write FLanguage;
    function      GetLanguageFile: string;
    function      GetFilesSizeUnit(const ByteString: string): string;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

implementation

uses
  Windows, Sysutils, Graphics,

  tb2toolbar,

  functions_sys, functions_str, functions_xml, Global;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

constructor TFolder.Create;
begin
  PathType := ftDefault;
  Path := '';
  inherited;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

destructor TFolder.Destroy;
begin
  inherited;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TFolder.UpdateFolder(const newPath: string);
begin
  if PathType = ftLast then
    Path := newPath;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

function TFolder.ReturnPath: string;
begin
  if DirectoryExists(Path) then
    Result := Path
  else
    Result := '';
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TFolder.SetFolder(const newPath: string; const newType: integer);
begin
  PathType := newType;
  case PathType of
     ftDefault : Path := '';
     ftUser : Path := newPath;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

const strXMLFolders: array [0..8] of string =
    ( 'Catalogs', 'Picture', 'Export', 'Templates', 'Import',
      'GetFromFiles', 'Graphic', 'Scripts', 'CustomFields' );

procedure TFolder.Load(Root: TJvSimpleXmlElem; nr: Integer);
begin
  Root := Root.Items.ItemNamed[strXMLFolders[nr]];
  if Root <> nil then
    with Root do
    begin
      PathType := Properties.IntValue('Type', ftLast);
      Path := Properties.Value('Path', '');
    end
  else
  begin
    PathType := ftLast;
    Path := '';
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TFolder.Load(ini: TMemIniFile; nr: Integer);
begin
  with ini do
  begin
    PathType := ReadInteger('Folders', 'Type'+intToStr(nr), ftLast);
    Path := ReadString('Folders', 'Path'+intToStr(nr), '');
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TFolder.Save(Items: TJvSimpleXmlElems; const nr: integer);
begin
  with Items.Add(strXMLFolders[nr]) do
  begin
    Properties.Add('Type', PathType);
    Properties.Add('Path', Path);
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

constructor TSettings.Create;
var
  i: Integer;
begin
  FXml := TJvSimpleXml.Create(nil);
  with rOptions do
  begin
    for i := 0 to Length(rFolders)-1 do
      rFolders[i] := TFolder.Create;
    for i := 0 to Length(rMovieInformation.rCombo)-1 do
      rMovieInformation.rCombo[i].Contents := TStringList.Create;
    rMovieList.Prefixes := TStringList.Create;
    rMovieInformation.SearchSites := TStringList.Create;
    rMovieInformation.rDefaultMovie.Values := TMovie.Create(nil);
  end;
  with rExport do
  begin
    HTMLMRU := TStringList.Create;
    CSVFields := TStringList.Create;
    SQLFields := TStringList.Create;
    OrderFields := TStringList.Create;
  end;
  with rImport do
  begin
    rCSV.Fields := TStringList.Create;
    LastFiles := TStringList.Create;
  end;
  with rRenumber do
  begin
    OrderFields := TStringList.Create;
  end;
  with rMain do
  begin
    MRU := TStringList.Create;
  end;
  with rScripts do
  begin
    ScriptMRU := TStringList.Create;
  end;
  with rPrint do
  begin
    OrderFields := TStringList.Create;
  end;
  inherited Create;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

destructor TSettings.Destroy;
var
  i: integer;
begin
  with rOptions do
  begin
    for i := 0 to Length(rFolders)-1 do
      rFolders[i].Free;
    for i := 0 to Length(rMovieInformation.rCombo)-1 do
      rMovieInformation.rCombo[i].Contents.Free;
    rMovieInformation.rDefaultMovie.Values.Free;
    rMovieInformation.SearchSites.Free;
    rMovieList.Prefixes.Free;
  end;
  with rExport do
  begin
    HTMLMRU.Free;
    CSVFields.Free;
    SQLFields.Free;
    OrderFields.Free;
  end;
  with rImport do
  begin
    rCsv.Fields.Free;
    LastFiles.Free;
  end;
  with rRenumber do
  begin
    OrderFields.Free;
  end;
  with rMain do
  begin
    MRU.Free;
  end;
  with rScripts do
  begin
    ScriptMRU.Free;
  end;
  with rPrint do
  begin
    OrderFields.Free;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TSettings.Load;
var
  i, c: integer;
  s: string;
  CurItem: TJvSimpleXmlElem;
  function GetCurItem(const ItemName: string): TJvSimpleXmlElem;
  begin
    Result := CurItem.Items.ItemNamed[ItemName];
    if Result = nil then
    begin
      Result := CurItem.Items.Add(ItemName);
      Result.Parent := CurItem;
    end;
  end;
  procedure LoadList(Items: TJvSimpleXmlElems; const PropName: string; List: TStrings; KeepItemName: Boolean = False);
  var
    i: Integer;
  begin
    List.Clear;
    with Items do
      for i := 0 to Count-1 do
        if KeepItemName then
          List.Values[Item[i].Name] := Item[i].Properties.ItemNamed[PropName].Value
        else
          List.Add(Item[i].Properties.ItemNamed[PropName].Value);
  end;
begin
  SetCurrentDir(strDirApp);
  c := ParamCount;
  for i := 1 to c do
  begin
    s := ParamStr(i);
    // if specified, loads another file than the normal one
    if SameText('/prefs', s) and (i < c) then
      strFileSettings := ParamStr(i + 1)
    // if specified, show errors when applying translation file
    else
    if SameText('/tr_show_errors', s) then
      Translator.ShowErrors := True;
  end;
  if not FileExists(strFileSettings) then
  begin
    // in old versions the file was moviecatalog.xml instead of prefs.xml
    if FileExists('moviecatalog.xml') then
      RenameFile(strDirApp + 'moviecatalog.xml', strFileSettings)
    else // load default settings, as no config file was found
      CopyFile(PChar(strFileDefault), PChar(strFileSettings), True);
  end;
  try
    FXml.LoadFromFile(strFileSettings);
  except
  end;
  CurItem := FXml.Root;
  version := FXml.Root.Properties.Value('Version', strVersion);
  CurItem := GetCurItem('Settings');
  CurItem := GetCurItem('Main');
  with CurItem, rMain do
  begin
    CurItem := GetCurItem('MRU');
    with CurItem do
    begin
      LoadList(Items, 'Name', MRU);
      CurItem := Parent;
    end;
    WindowWidth := Properties.IntValue('WindowWidth', 750);
    WindowHeight := Properties.IntValue('WindowHeight', 550);
    WindowLeft := Properties.IntValue('WindowLeft', 20);
    WindowTop := Properties.IntValue('WindowTop', 20);
    WindowState := Properties.IntValue('WindowState', 2);
    Statusbar := Properties.BoolValue('Statusbar', True);
    ListWidth := Properties.IntValue('ListWidth', 200);
    if ListWidth < 1 then
      ListWidth := 200;
    GridMode := Properties.BoolValue('GridMode', False);
    ListAlClient := Properties.BoolValue('ListAlClient', False);
    DisplayHTML := Properties.BoolValue('DisplayHTML', False);
    DisplayThumbnails := Properties.BoolValue('DisplayThumbnails', False);
    PictureDockedHeight := Properties.IntValue('PicDockedHeight', 180);
    Group := Properties.IntValue('Group', -1);
    FindWholefield := Properties.BoolValue('FindWholeField', False);
    FindField := Properties.IntValue('FindField', -1);
    SortField := Properties.IntValue('SortField', 1); // fieldNumber + 1
    ColumnSettings := Properties.Value('ColumnSettings', '');
    PictureWinLeft := Properties.IntValue('PictureWinLeft', 100);
    PictureWinTop := Properties.IntValue('PictureWinTop', 100);
    Toolbars := Items.ItemNamed['Toolbars'];
    if Toolbars = nil then
      Toolbars := Items.Add('Toolbars');
    CurItem := Parent;
  end; // Main
  CurItem := GetCurItem('Export');
  with CurItem, rExport do
  begin
    WindowWidth := Properties.IntValue('WindowWidth', 772);
    WindowHeight := Properties.IntValue('WindowHeight', 510);
    WindowState := Properties.IntValue('WindowState', 1);
    Includemov := Properties.IntValue('Includemov', 0);
    ImgFileName := Properties.IntValue('ImgFileName', 0);
    CurItem := GetCurItem('HTML');
    with CurItem do
    begin
      ExportBoth := Properties.BoolValue('ExportBoth', True);
      HTMLLastTemplate := Properties.Value('LastTemplate', strDirApp + strDirTemplates + 'template.html');
      HTMLLastTemplate2 := Properties.Value('LastTemplate2', '');
      CurItem := GetCurItem('MRU');
      with CurItem do
      begin
        LoadList(Items, 'Name', HTMLMRU);
        CurItem := Parent;
      end;
      CurItem := Parent;
    end; // Export/HTML
    CurItem := GetCurItem('CSV');
    with CurItem do
    begin
      CSVDelimiter := Properties.Value('Delimiter', ',');
      CSVBloc := Properties.Value('Bloc', '"');
      CSVLinebreaks := Properties.Value('Linebreaks', ' ');
      CSVColumnTitles := Properties.BoolValue('ColumnTitles', False);
      CurItem := GetCurItem('Fields');
      with CurItem do
      begin
        LoadList(Items, 'Name', CSVFields);
        CurItem := Parent;
      end;
      CurItem := Parent;
    end; // Export/CSV
    CurItem := GetCurItem('SQL');
    with CurItem do
    begin
      SQLUpdate := Properties.BoolValue('SQLUpdate', False);
      SQLDrop := Properties.BoolValue('SQLDrop', True);
      SQLCreate := Properties.BoolValue('SQLCreate', True);
      CurItem := GetCurItem('Fields');
      with CurItem do
      begin
        LoadList(Items, 'Name', SQLFields);
        CurItem := Parent;
      end;
      SQLTableName := Properties.Value('SQLTableName', 'movies');
      SQLLinebreaks := Properties.Value('SQLLinebreaks', '\n');
      CurItem := Parent;
    end; // Export/SQL
    CurItem := GetCurItem('Sort');
    with CurItem do
    begin
      OrderBy := Properties.IntValue('Order', 0);
      CurItem := GetCurItem('Fields');
      with CurItem do
      begin
        LoadList(Items, 'Name', OrderFields);
        CurItem := Parent;
      end;
      CurItem := Parent;
    end; // Export/Sort
    CurItem := Parent;
  end; // Export
  CurItem := GetCurItem('Import');
  with CurItem, rImport do
  begin
    WindowWidth := Properties.IntValue('WindowWidth', 780);
    WindowHeight := Properties.IntValue('WindowHeight', 520);
    WindowState := Properties.IntValue('WindowState', 1);
    AllowDup := Properties.BoolValue('AllowDup', True);
    AutoAssign := Properties.BoolValue('AutoAssign', True);
    PicImport := Properties.IntValue('PicImport', 0);
    CurItem := GetCurItem('CSV');
    with CurItem, rCsv do
    begin
      Delim := Properties.Value('Delim', '[tab]');
      Quote := Properties.Value('Quote', '');
      FirstLineHeaders := Properties.BoolValue('FirstLineHeaders', False);
      Linebreaks := Properties.Value('Linebreaks', '');
      CurItem := GetCurItem('Fields');
      with CurItem do
      begin
        LoadList(Items, 'Name', Fields);
        CurItem := Parent;
      end;
      CurItem := Parent;
    end;
    CurItem := GetCurItem('Query');
    with CurItem, rQuery do
    begin
      From := Properties.Value('From', '');
      Where := Properties.Value('Where', '');
      CurItem := Parent;
    end;
    CurItem := GetCurItem('LastFiles');
    with CurItem do
    begin
      LoadList(Items, 'FileName', LastFiles, True);
      CurItem := Parent;
    end;
    CurItem := Parent;
  end; // Import
  CurItem := GetCurItem('Print');
  with CurItem, rPrint do
  begin
    WindowWidth := Properties.IntValue('WindowWidth', 750);
    WindowHeight := Properties.IntValue('WindowHeight', 480);
    WindowState := Properties.IntValue('WindowState', 1);
    CurItem := GetCurItem('Sort');
    with CurItem do
    begin
      OrderBy := Properties.IntValue('Order', 0);
      CurItem := GetCurItem('Fields');
      with CurItem do
      begin
        LoadList(Items, 'Name', OrderFields);
        CurItem := Parent;
      end;
      CurItem := Parent;
    end; // Print/Sort
    Includemov := Properties.IntValue('Includemov', 0);
    ReportsListWidth := Properties.IntValue('ReportsListWidth', 175);
    if ReportsListWidth < 1 then
      ReportsListWidth := 175;
    CurItem := Parent;
  end; // Print
  CurItem := GetCurItem('Scripts');
  with CurItem, rScripts do
  begin
    WindowWidth := Properties.IntValue('WindowWidth', 580);
    WindowHeight := Properties.IntValue('WindowHeight', 430);
    WindowState := Properties.IntValue('WindowState', 1);
    ScriptRemember1 := Properties.Value('ScriptRemember1', '');
    ScriptRemember2 := Properties.Value('ScriptRemember2', '');
    Includemov := Properties.IntValue('Includemov', 0);
    LangFilterExcl := Properties.Value('LangFilterExcl', '');
    ListViewStyle := Properties.IntValue('ListViewStyle', 2 {vsList});
    ShowResGetInfo := Properties.BoolValue('ShowResGetInfo', True);
    ShowResScripting := Properties.BoolValue('ShowResScripting', False);
    CloseWinGetInfo := Properties.BoolValue('CloseWinGetInfo', True);
    CloseWinScripting := Properties.BoolValue('CloseWinScripting', False);
    AllowClearGetInfo := Properties.BoolValue('AllowClearGetInfo', False);
    AllowClearScripting := Properties.BoolValue('AllowClearScripting', True);
    CurItem := GetCurItem('MRU');
    with CurItem do
    begin
      LoadList(Items, 'Name', ScriptMRU);
      CurItem := Parent;
    end;
    CurItem := GetCurItem('Properties');
    with CurItem, rProperties do
    begin
      WindowWidth := Properties.IntValue('WindowWidth', 500);
      WindowHeight := Properties.IntValue('WindowHeight', 450);
      CurItem := Parent;
    end;
    CurItem := GetCurItem('Results');
    with CurItem, rResults do
    begin
      WindowWidth := Properties.IntValue('WindowWidth', 690);
      WindowHeight := Properties.IntValue('WindowHeight', 550);
      CurItem := Parent;
    end;
    CurItem := GetCurItem('PickList');
    with CurItem, rPickList do
    begin
      WindowWidth := Properties.IntValue('WindowWidth', 500);
      WindowHeight := Properties.IntValue('WindowHeight', 440);
      CurItem := Parent;
    end;
    CurItem := GetCurItem('PickTree');
    with CurItem, rPickTree do
    begin
      WindowWidth := Properties.IntValue('WindowWidth', 450);
      WindowHeight := Properties.IntValue('WindowHeight', 440);
      CurItem := Parent;
    end;
    Toolbars := Items.ItemNamed['Toolbars'];
    if Toolbars = nil then
      Toolbars := Items.Add('Toolbars');
    CurItem := Parent;
  end; // Scripts
  CurItem := GetCurItem('Loan');
  with CurItem, rLoan do
  begin
    WindowWidth := Properties.IntValue('WindowWidth', 650);
    WindowHeight := Properties.IntValue('WindowHeight', 550);
    WindowState := Properties.IntValue('WindowState', 1);
    LvMoviesHeight := Properties.IntValue('MoviesHeight', 175);
    if LvMoviesHeight < 1 then
      LvMoviesHeight := 175;
    LvMoviesColWidth := Properties.IntValue('MoviesColWidth', WindowWidth div 3);
    LvMoviesSort := Properties.IntValue('MoviesSort', 0);
    LvNamesWidth := Properties.IntValue('NamesWidth', 210);
    if LvNamesWidth < 1 then
      LvNamesWidth := 210;
    LvNamesSort := Properties.IntValue('NamesSort', 0);
    LvLentSort := Properties.IntValue('LentSort', 0);
    IncludeSameNum := Properties.BoolValue('IncludeSameNum', False);
    IncludeSameLabel := Properties.BoolValue('IncludeSameLabel', False);
    CurItem := Parent;
  end; // Loan
  CurItem := GetCurItem('Properties');
  with CurItem, rProperties do
  begin
    WindowWidth := Properties.IntValue('WindowWidth', 400);
    WindowHeight := Properties.IntValue('WindowHeight', 350);
    CurItem := Parent;
  end; // Properties
  CurItem := GetCurItem('Statistics');
  with CurItem, rStatistics do
  begin
    WindowWidth := Properties.IntValue('WindowWidth', 750);
    WindowHeight := Properties.IntValue('WindowHeight', 550);
    WindowState := Properties.IntValue('WindowState', 1);
    EmptyMonths := Properties.BoolValue('EmptyMonths', True);
    Legend := Properties.BoolValue('Legend', True);
    Labels := Properties.BoolValue('Labels', False);
    Group := Properties.BoolValue('GroupValues', True);
    Includemov := Properties.IntValue('Includemov', 0);
    CurItem := Parent;
  end; // Statistics
  CurItem := GetCurItem('Renumber');
  with CurItem, rRenumber do
  begin
    OrderBy := Properties.IntValue('SortOrder', 0);
    CurItem := GetCurItem('Fields');
    with CurItem do
    begin
      LoadList(Items, 'Name', OrderFields);
      CurItem := Parent;
    end;
    CurItem := Parent;
  end; // Renumber
  CurItem := GetCurItem('Options');
  with CurItem, rOptions do
  begin
    WindowWidth := Properties.IntValue('WindowWidth', 650);
    WindowHeight := Properties.IntValue('WindowHeight', 500);
    CurItem := GetCurItem('Display');
    with CurItem, rDisplay do
    begin
      ImagesInMenu := Properties.BoolValue('ImagesInMenu', True);
      OfficeXP := Properties.BoolValue('OfficeXP', False);
      if IsThemedXP then
        IconSet := Properties.Value('IconSet', 'Windows XP')
      else
        IconSet := Properties.Value('IconSet', 'Scrows');
      if (IconSet = '') and (StrToIntDef(StringReplace(Copy(version, 1, 3), '.', '', []), 35) < 35) then
        IconSet := 'Scrows';
      ColorType := Properties.IntValue('ColorType', 0);
      Logo := Properties.BoolValue('Logo', True);
      SoftBorders := Properties.BoolValue('SoftBorders', False);
      CurItem := Parent;
    end; // Options/Display
    CurItem := GetCurItem('Files');
    with CurItem, rFiles do
    begin
//          XMLHeader := Properties.Value('XMLHeader', 'iso-8859-1');
      Backup := Properties.BoolValue('Backup', True);
      RecentFiles := Properties.IntValue('RecentFiles', 5);
      CurItem := GetCurItem('AutoLoad');
      with CurItem do
      begin
        AutoLoad := Properties.BoolValue('Enabled', False);
        AutoLoadFile := Properties.Value('File', '');
        AutoLoadLast := Properties.BoolValue('Last', False);
        CurItem := Parent;
      end; // Options/Files/Autoload
      CurItem := GetCurItem('AutoLoadCF');
      with CurItem do
      begin
        AutoLoadCF := Properties.BoolValue('Enabled', False);
        AutoLoadCFFile := Properties.Value('File', '');
        CurItem := Parent;
      end; // Options/Files/AutoloadCF
      CurItem := GetCurItem('History');
      with CurItem do
      begin
        History := Properties.BoolValue('Enabled', True);
        HistoryFile := Properties.Value('File', 'Loans history.csv');
        CurItem := Parent;
      end; // Options/Files/History
      CurItem := GetCurItem('HTMLTemplate');
      with CurItem do
      begin
        HTMLTemplateFile := Properties.Value('File', 'HTMLDefaultTemplate.html');
        CurItem := Parent;
      end; // Options/Files/HTMLTemplate
      CurItem := Parent;
    end; // Options/Files
    CurItem := GetCurItem('MovieList');
    with CurItem, rMovieList do
    begin
      CurItem := GetCurItem('TitleDisplay');
      with CurItem do
      begin
        TitleColumn := Properties.IntValue('TitleColumn', 2);
        CurItem := GetCurItem('Prefixes');
        with CurItem do
        begin
          UsePrefixes := Properties.BoolValue('Use', False);
          LoadList(Items, 'Text', Prefixes);
          CurItem := Parent;
        end;
        CurItem := Parent;
      end; // Options/MovieList/TitleDisplay
      ConfirmDelete := Properties.BoolValue('ConfirmDelete', True);
      ConfirmUndo := Properties.BoolValue('ConfirmUndo', True);
      Checkboxes := Properties.BoolValue('Checkboxes', True);
      HotTrack := Properties.BoolValue('HotTrack', False);
      EnhancedScrollbars := Properties.BoolValue('EnhancedScrollbars', False);
      MovieNumColumn := Properties.BoolValue('MovieNumColumn', True);
      GridTextSize := Properties.IntValue('GridTextSize', 32);
      ShortcutPrev := Properties.IntValue('ShortcutPrev', 16417);
      ShortcutNext := Properties.IntValue('ShortcutNext', 16418);
      GroupCount := Properties.BoolValue('GroupCount', True);
      GroupExpand := Properties.BoolValue('GroupExpand', False);
      GroupsAbove := Properties.BoolValue('GroupsAbove', False);
      GroupUnique := Properties.BoolValue('GroupUnique', True);
      GroupMulti := Properties.BoolValue('GroupMulti', True);
      GroupMultiSep := Properties.Value('GroupMultiSep', ',')[1];
      GroupMultiRmAllP := Properties.BoolValue('GroupMultiRmAllP', False);
      GroupMultiAddPatch := Properties.BoolValue('GroupMultiAddPatch', True);
      SortGroupsByCount := Properties.BoolValue('SortGroupsByCount', False);
      CurItem := Parent;
    end; // Options/MovieList
    CurItem := GetCurItem('MovieInfo');
    with CurItem, rMovieInformation do
    begin
      AskNumber := Properties.BoolValue('AskNumber', True);
      FirstAvailable := Properties.BoolValue('FirstAvailable', False);
      AddScript := Properties.BoolValue('AddScript', False);
      AddFiles := Properties.BoolValue('AddFiles', False);
      RatingTrunc := Properties.BoolValue('RatingTrunc', False);
      SetCurrentDate := Properties.BoolValue('SetCurrentDate', True);
      CurItem := GetCurItem('Picture');
      with CurItem do
      begin
        PictureBackground := Properties.IntValue('Background', clDefault);
        PictureFitWindow := Properties.BoolValue('FitWindow', True);
        PictureInfo := Properties.BoolValue('PictureInfo', False);
        CurItem := Parent;
      end; // Options/MovieInfo/Picture
      CurItem := GetCurItem('DefaultMovie');
      with CurItem, rDefaultMovie do
      begin
        WindowWidth := Properties.IntValue('WindowWidth', 570);
        WindowHeight := Properties.IntValue('WindowHeight', 560);
        Values.InitFields;
        try
          if (Items.ItemNamed['Values'] <> nil) then
            Values.LoadFromXML(Items.ItemNamed['Values']);
        except
        end;
        CurItem := Parent;
      end; // Options/MovieInfo/DefaultValues
      CurItem := GetCurItem('ImportInfo');
      with CurItem do
      begin
        ImportMediaLabel := Properties.BoolValue('MediaLabel', False);
        ImportFileName := Properties.BoolValue('FileName', False);
        ImportFileNameURL := Properties.BoolValue('FileNameURL', False);
        ImportSize := Properties.BoolValue('Size', True);
        ImportSizeString := Properties.BoolValue('SizeString', True);
        ImportSizeUnit := TFileSizeUnit(Properties.IntValue('SizeUnit', 2));
        if StrToIntDef(StringReplace(Copy(version, 1, 3), '.', '', []), 35) < 35 then
        // unit "GB" added to the list in version 3.5, and the list order was reversed
          case Integer(ImportSizeUnit) of
            1:  ImportSizeUnit := fsuKB;
            2:  ImportSizeUnit := fsuB;
          else
            ImportSizeUnit := fsuMB;
          end;
        ImportInternalAVI := Properties.BoolValue('InternalAVI', False);
        ImportResolution := Properties.BoolValue('Resolution', True);
        ImportLength := Properties.BoolValue('Length', True);
        ImportFramerate := Properties.BoolValue('Framerate', True);
        ImportVideoCodec := Properties.BoolValue('VideoCodec', True);
        ImportVideoBitrate := Properties.BoolValue('VideoBitrate', True);
        ImportAudioCodec := Properties.BoolValue('AudioCodec', True);
        ImportAudioBitrate := Properties.BoolValue('AudioBitrate', True);
        ImportLanguages := Properties.BoolValue('Languages', True);
        ImportSubtitles := Properties.BoolValue('Subtitles', True);
        CurItem := Parent;
      end; // Options/MovieInfo/ImportInfo
      CurItem := GetCurItem('PicImport');
      with CurItem, rPicImport do
      begin
        GetInfoMethod := Properties.IntValue('GetInfoMethod', 0);
        ScriptingMethod := Properties.IntValue('ScriptingMethod', 1);
        Naming := Properties.IntValue('Naming', 1);
        CatalogPrefix := Properties.BoolValue('CatalogPrefix', True);
        CurItem := Parent;
      end; // Options/MovieInfo/PicImport
      CurItem := GetCurItem('Search');
      with CurItem do
      begin
        LoadList(Items, 'Address', SearchSites);
        CurItem := Parent;
      end;
      CurItem := GetCurItem('Lists');
      with CurItem do
      begin
        ComboSameForAll := Properties.BoolValue('SameForAll', False);
        for i := 0 to Length(rCombo)-1 do
        begin
          CurItem := GetCurItem(strXMLDdl[i]);
          with CurItem, rCombo[i] do
          begin
            AutoAdd := Properties.BoolValue('AutoAdd', False);
            Sort := Properties.BoolValue('Sorted', True);
            AutoComplete := Properties.BoolValue('AutoComplete', False);
            UseCatalogValues := Properties.BoolValue('UseCatalogValues', True);
            LoadList(Items, 'Text', Contents);
            CurItem := Parent;
          end;
        end;
        CurItem := Parent;
      end;
      CurItem := Parent;
    end; // Options/MovieInfo
    CurItem := GetCurItem('Scripting');
    with CurItem, rScripting do
    begin
      ScriptAutorun := Properties.BoolValue('Autorun', False);
      CurItem := GetCurItem('Proxy');
      with CurItem do
      begin
        Proxy := Properties.BoolValue('Use', False);
        ProxyServer := Properties.Value('Server', '');
        ProxyPort := Properties.IntValue('Port', 8080);
        ProxyUsername := Properties.Value('Username', '');
        ProxyPassword := Decrypt(Properties.Value('Password', ''));
        KeepConnection := Properties.BoolValue('KeepConnection', False);
        HTTP10 := Properties.BoolValue('HTTP10', False);
        CurItem := Parent;
      end;
      CurItem := Parent;
    end; // Options/Scripting
    CurItem := GetCurItem('Export');
    with CurItem, rExport do
    begin
      LoadTemplate := Properties.BoolValue('LoadTemplate', True);
      Linebreak := Properties.Value('LineBreak', '<br />');
      ForcePicSizeW := Properties.IntValue('ForcePicSizeW', -1);
      ForcePicSizeH := Properties.IntValue('ForcePicSizeH', -1);
      SQLDate := Properties.Value('SQLDate', 'yyyy''-''mm''-''dd');
      RememberLastFile := Properties.BoolValue('RememberLastFile', False);
      LastFile := Properties.Value('LastFile', '');
      OpenExportedFile := Properties.BoolValue('OpenExportedFile', False);
      ExpFileStyle := Properties.IntValue('ExpFileStyle', 0);
      ExpFileExt := Properties.Value('ExpFileExt', '');
      AddZeroes := Properties.BoolValue('AddZeroes', False);
      CopyPictures := Properties.BoolValue('CopyPictures', True);
      CopyPicturesNew := Properties.BoolValue('CopyPicturesNew', False);
      CurItem := Parent;
    end; // Options/Export
    CurItem := GetCurItem('Folders');
    with CurItem do
    begin
      SameFolderForAll := Properties.BoolValue('SameForAll', False);
      for i := 0 to length(rFolders)-1 do
        with rFolders[i] do
        begin
          Load(CurItem, i);
          if (FolderType <> ftDefault) and (Value = '') then
          begin
            Value := strDirApp;
            case i of
              fuCatalogs,
              fuExport,
              fuImport,
              fuPicture:    Value := Value + strDirCatalogs;
              fuTemplates:  Value := Value + strDirTemplates;
              fuScripts:    Value := Value + strDirScripts;
              fuCustomFields: Value := Value + strDirCatalogs;
            end;
          end;
        end; // for-with
      CurItem := Parent;
    end; // Options/Folders
    CurItem := GetCurItem('Language');
    with CurItem, rLanguage do
    begin
      Language := Properties.Value('File', '?');
      CurItem := Parent;
    end; // Options/Languages
    CurItem := Parent;
  end; // Options
  CurItem := GetCurItem('CustomFieldsManager');
  with CurItem, rCustomFieldsManager do
  begin
    WindowWidth := Properties.IntValue('WindowWidth', 450);
    WindowHeight := Properties.IntValue('WindowHeight', 350);
    CurItem := Parent;
  end; // CustomFieldsManager
  version := ConstValues.strVersion;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TSettings.Save;
var
  i: integer;
  function PrepareItem(Items: TJvSimpleXmlElems; const AName: string): TJvSimpleXmlElem;
  begin
    Result := Items.ItemNamed[AName];
    if Result = nil then
      Result := Items.Add(AName)
    else
      Result.Properties.Clear;
  end;
  procedure SaveList(Items: TJvSimpleXmlElems; const SubItemName, PropName: string; List: TStrings);
  var
    i: Integer;
  begin
    Items.Clear;
    for i := 0 to List.Count-1 do
      if SubItemName = '' then
        with Items.Add(List.Names[i]) do
          Properties.Add(PropName, List.ValueFromIndex[i])
      else
        with Items.Add(SubItemName) do
          Properties.Add(PropName, List.Strings[i]);
  end;
begin
  WriteXMLHeader(FXml.Root, intFileVersion, 'AntMovieCatalog', strVersion, strDate);
  with PrepareItem(FXml.Root.Items, 'Settings') do
  begin
    with PrepareItem(Items, 'Main'), rMain do
    begin
      SaveList(PrepareItem(Items, 'MRU').Items, 'File', 'Name', MRU);
      Properties.Add('WindowWidth', WindowWidth);
      Properties.Add('WindowHeight', WindowHeight);
      Properties.Add('WindowLeft', WindowLeft);
      Properties.Add('WindowTop', WindowTop);
      Properties.Add('WindowState', WindowState);
      Properties.Add('Statusbar', Statusbar);
      Properties.Add('ListWidth', ListWidth);
      Properties.Add('GridMode', GridMode);
      Properties.Add('ListAlClient', ListAlClient);
      Properties.Add('DisplayHTML', DisplayHTML);
      Properties.Add('DisplayThumbnails', DisplayThumbnails);
      Properties.Add('PicDockedHeight', PictureDockedHeight);
      Properties.Add('Group', Group);
      Properties.Add('FindWholeField', FindWholefield);
      Properties.Add('FindField', FindField);
      Properties.Add('SortField', SortField);
      Properties.Add('ColumnSettings', ColumnSettings);
      Properties.Add('PictureWinLeft', PictureWinLeft);
      Properties.Add('PictureWinTop', PictureWinTop);
    end; // Main
    with PrepareItem(Items, 'Export'), rExport do
    begin
      Properties.Add('WindowWidth', WindowWidth);
      Properties.Add('WindowHeight', WindowHeight);
      Properties.Add('WindowState', WindowState);
      Properties.Add('Includemov', Includemov);
      Properties.Add('ImgFileName', ImgFileName);
      with PrepareItem(Items, 'HTML') do
      begin
        Properties.Add('ExportBoth', ExportBoth);
        Properties.Add('LastTemplate', HTMLLastTemplate);
        Properties.Add('LastTemplate2', HTMLLastTemplate2);
        SaveList(PrepareItem(Items, 'MRU').Items, 'File', 'Name', HTMLMRU);
      end; // Export/HTML
      with PrepareItem(Items, 'CSV') do
      begin
        Properties.Add('Delimiter', CSVDelimiter);
        Properties.Add('Bloc', CSVBloc);
        Properties.Add('Linebreaks', CSVLinebreaks);
        Properties.Add('ColumnTitles', CSVColumnTitles);
        SaveList(PrepareItem(Items, 'Fields').Items, 'Field', 'Name', CSVFields);
      end; // Export/CSV
      with PrepareItem(Items, 'SQL') do
      begin
        Properties.Add('SQLUpdate', SQLUpdate);
        Properties.Add('SQLDrop', SQLDrop);
        Properties.Add('SQLCreate', SQLCreate);
        SaveList(PrepareItem(Items, 'Fields').Items, 'Field', 'Name', SQLFields);
        Properties.Add('SQLTableName', SQLTableName);
        Properties.Add('SQLLinebreaks', SQLLinebreaks);
      end; // Export/SQL
      with PrepareItem(Items, 'Sort') do
      begin
        Properties.Add('Order', OrderBy);
        SaveList(PrepareItem(Items, 'Fields').Items, 'Field', 'Name', OrderFields);
      end; // Export/Sort
    end; // Export
    with PrepareItem(Items, 'Import'), rImport do
    begin
      Properties.Add('WindowWidth', WindowWidth);
      Properties.Add('WindowHeight', WindowHeight);
      Properties.Add('WindowState', WindowState);
      Properties.Add('AllowDup', AllowDup);
      Properties.Add('AutoAssign', AutoAssign);
      Properties.Add('PicImport', PicImport);
      with PrepareItem(Items, 'CSV'), rCsv do
      begin
        Properties.Add('Delim', Delim);
        Properties.Add('Quote', Quote);
        Properties.Add('FirstLineHeaders', FirstLineHeaders);
        Properties.Add('Linebreaks', Linebreaks);
        SaveList(PrepareItem(Items, 'Fields').Items, 'Field', 'Name', Fields);
      end;
      with PrepareItem(Items, 'Query'), rQuery do
      begin
        Properties.Add('From', From);
        Properties.Add('Where', Where);
      end;
      SaveList(PrepareItem(Items, 'LastFiles').Items, '', 'FileName', LastFiles);
    end; // Import
    with PrepareItem(Items, 'Print'), rPrint do
    begin
      Properties.Add('WindowWidth', WindowWidth);
      Properties.Add('WindowHeight', WindowHeight);
      Properties.Add('WindowState', WindowState);
      with PrepareItem(Items, 'Sort') do
      begin
        Properties.Add('Order', OrderBy);
        SaveList(PrepareItem(Items, 'Fields').Items, 'Field', 'Name', OrderFields);
      end; // Print/Sort
      Properties.Add('Includemov', Includemov);
      Properties.Add('ReportsListWidth', ReportsListWidth);
    end; // Print
    with PrepareItem(Items, 'Scripts'), rScripts do
    begin
      Properties.Add('WindowWidth', WindowWidth);
      Properties.Add('WindowHeight', WindowHeight);
      Properties.Add('WindowState', WindowState);
      Properties.Add('ScriptRemember1', ScriptRemember1);
      Properties.Add('ScriptRemember2', ScriptRemember2);
      Properties.Add('Includemov', Includemov);
      Properties.Add('LangFilterExcl', LangFilterExcl);
      Properties.Add('ListViewStyle', ListViewStyle);
      Properties.Add('ShowResGetInfo', ShowResGetInfo);
      Properties.Add('ShowResScripting', ShowResScripting);
      Properties.Add('CloseWinGetInfo', CloseWinGetInfo);
      Properties.Add('CloseWinScripting', CloseWinScripting);
      SaveList(PrepareItem(Items, 'MRU').Items, 'File', 'Name', ScriptMRU);
      with PrepareItem(Items, 'Properties'), rProperties do
      begin
        Properties.Add('WindowWidth', WindowWidth);
        Properties.Add('WindowHeight', WindowHeight);
      end;
      with PrepareItem(Items, 'Results'), rResults do
      begin
        Properties.Add('WindowWidth', WindowWidth);
        Properties.Add('WindowHeight', WindowHeight);
      end;
      with PrepareItem(Items, 'PickList'), rPickList do
      begin
        Properties.Add('WindowWidth', WindowWidth);
        Properties.Add('WindowHeight', WindowHeight);
      end;
      with PrepareItem(Items, 'PickTree'), rPickTree do
      begin
        Properties.Add('WindowWidth', WindowWidth);
        Properties.Add('WindowHeight', WindowHeight);
      end;
    end; // Scripts
    with PrepareItem(Items, 'Loan'), rLoan do
    begin
      Properties.Add('WindowWidth', WindowWidth);
      Properties.Add('WindowHeight', WindowHeight);
      Properties.Add('WindowState', WindowState);
      Properties.Add('MoviesHeight', LvMoviesHeight);
      Properties.Add('MoviesColWidth', LvMoviesColWidth);
      Properties.Add('MoviesSort', LvMoviesSort);
      Properties.Add('NamesWidth', LvNamesWidth);
      Properties.Add('NamesSort', LvNamesSort);
      Properties.Add('LentSort', LvLentSort);
      Properties.Add('IncludeSameNum', IncludeSameNum);
      Properties.Add('IncludeSameLabel', IncludeSameLabel);
    end; // Loan
    with PrepareItem(Items, 'Properties'), rProperties do
    begin
      Properties.Add('WindowWidth', WindowWidth);
      Properties.Add('WindowHeight', WindowHeight);
    end; // Properties
    with PrepareItem(Items, 'Statistics'), rStatistics do
    begin
      Properties.Add('WindowWidth', WindowWidth);
      Properties.Add('WindowHeight', WindowHeight);
      Properties.Add('WindowState', WindowState);
      Properties.Add('EmptyMonths', EmptyMonths);
      Properties.Add('Legend', Legend);
      Properties.Add('Labels', Labels);
      Properties.Add('GroupValues', Group);
      Properties.Add('Includemov', Includemov);
    end; // Statistics
    with PrepareItem(Items, 'Renumber'), rRenumber do
    begin
      Properties.Add('SortOrder', OrderBy);
      SaveList(PrepareItem(Items, 'Fields').Items, 'Field', 'Name', OrderFields);
    end; // Renumber
    with PrepareItem(Items, 'Options'), rOptions do
    begin
      Properties.Add('WindowWidth', WindowWidth);
      Properties.Add('WindowHeight', WindowHeight);
      with PrepareItem(Items, 'Display'), rDisplay do
      begin
        Properties.Add('ImagesInMenu', ImagesInMenu);
        Properties.Add('OfficeXP', OfficeXP);
        Properties.Add('IconSet', IconSet);
        Properties.Add('ColorType', ColorType);
        Properties.Add('Logo', Logo);
        Properties.Add('SoftBorders', SoftBorders);
      end; // Options/Display
      with PrepareItem(Items, 'Files'), rFiles do
      begin
//            Properties.Add('XMLHeader', XMLHeader);
        Properties.Add('Backup', Backup);
        Properties.Add('RecentFiles', RecentFiles);
        with PrepareItem(Items, 'AutoLoad') do
        begin
          Properties.Add('Enabled', AutoLoad);
          Properties.Add('File', AutoLoadFile);
          Properties.Add('Last', AutoLoadLast);
        end; // Options/Files/Autoload
        with PrepareItem(Items, 'AutoLoadCF') do
        begin
          Properties.Add('Enabled', AutoLoadCF);
          Properties.Add('File', AutoLoadCFFile);
        end; // Options/Files/AutoloadCF
        with PrepareItem(Items, 'History') do
        begin
          Properties.Add('Enabled', History);
          Properties.Add('File', HistoryFile);
        end; // Options/Files/History
        with PrepareItem(Items, 'HTMLTemplate') do
        begin
          Properties.Add('File', HTMLTemplateFile);
        end; // Options/Files/HTMLTemplate
      end; // Options/Files
      with PrepareItem(Items, 'MovieList'), rMovieList do
      begin
        with PrepareItem(Items, 'TitleDisplay') do
        begin
          Properties.Add('TitleColumn', TitleColumn);
          with PrepareItem(Items, 'Prefixes') do
          begin
            SaveList(Items, 'Item', 'Text', Prefixes);
            Properties.Add('Use', UsePrefixes);
          end; // Options/MovieList/TitleDisplay/Prefixes
        end; // Options/MovieList/TitleDisplay
        Properties.Add('ConfirmDelete', ConfirmDelete);
        Properties.Add('ConfirmUndo', ConfirmUndo);
        Properties.Add('Checkboxes', Checkboxes);
        Properties.Add('HotTrack', HotTrack);
        Properties.Add('EnhancedScrollbars', EnhancedScrollbars);
        Properties.Add('MovieNumColumn', MovieNumColumn);
        Properties.Add('GridTextSize', GridTextSize);
        Properties.Add('ShortcutPrev', ShortcutPrev);
        Properties.Add('ShortcutNext', ShortcutNext);
        Properties.Add('GroupCount', GroupCount);
        Properties.Add('GroupExpand', GroupExpand);
        Properties.Add('GroupsAbove', GroupsAbove);
        Properties.Add('GroupUnique', GroupUnique);
        Properties.Add('GroupMulti', GroupMulti);
        Properties.Add('GroupMultiSep', GroupMultiSep);
        Properties.Add('GroupMultiRmAllP', GroupMultiRmAllP);
        Properties.Add('GroupMultiAddPatch', GroupMultiAddPatch);
        Properties.Add('SortGroupsByCount', SortGroupsByCount);
      end; // Options/MovieList
      with PrepareItem(Items, 'MovieInfo'), rMovieInformation do
      begin
        Properties.Add('AskNumber', AskNumber);
        Properties.Add('FirstAvailable', FirstAvailable);
        Properties.Add('AddScript', AddScript);
        Properties.Add('AddFiles', AddFiles);
        Properties.Add('RatingTrunc', RatingTrunc);
        Properties.Add('SetCurrentDate', SetCurrentDate);
        with PrepareItem(Items, 'Picture') do
        begin
          Properties.Add('Background', PictureBackground);
          Properties.Add('FitWindow', PictureFitWindow);
          Properties.Add('PictureInfo', PictureInfo);
        end; // Options/MovieInfo/Picture
        with PrepareItem(Items, 'DefaultMovie'), rDefaultMovie do
        begin
          Properties.Add('WindowWidth', WindowWidth);
          Properties.Add('WindowHeight', WindowHeight);
          Values.SaveToXML(PrepareItem(Items, 'Values'));
        end; // Options/MovieInfo/DefaultMovie
        with PrepareItem(Items, 'ImportInfo') do
        begin
          Properties.Add('MediaLabel', ImportMediaLabel);
          Properties.Add('FileName', ImportFileName);
          Properties.Add('FileNameURL', ImportFileNameURL);
          Properties.Add('Size', ImportSize);
          Properties.Add('SizeString', ImportSizeString);
          Properties.Add('SizeUnit', Integer(ImportSizeUnit));
          Properties.Add('InternalAVI', ImportInternalAVI);
          Properties.Add('Resolution', ImportResolution);
          Properties.Add('Length', ImportLength);
          Properties.Add('Framerate', ImportFramerate);
          Properties.Add('VideoCodec', ImportVideoCodec);
          Properties.Add('VideoBitrate', ImportVideoBitrate);
          Properties.Add('AudioCodec', ImportAudioCodec);
          Properties.Add('AudioBitrate', ImportAudioBitrate);
          Properties.Add('Languages', ImportLanguages);
          Properties.Add('Subtitles', ImportSubtitles);
        end; // Options/MovieInfo/ImportInfo
        with PrepareItem(Items, 'PicImport'), rPicImport do
        begin
          Properties.Add('GetInfoMethod', GetInfoMethod);
          Properties.Add('ScriptingMethod', ScriptingMethod);
          Properties.Add('Naming', Naming);
          Properties.Add('CatalogPrefix', CatalogPrefix);
        end; // Options/MovieInfo/PicImport
        SaveList(PrepareItem(Items, 'Search').Items, 'Site', 'Address', SearchSites);
        with PrepareItem(Items, 'Lists') do
        begin
          Properties.Add('SameForAll', ComboSameForAll);
          for i := 0 to Length(rCombo)-1 do
          begin
            with PrepareItem(Items, strXMLDdl[i]), rCombo[i] do
            begin
              SaveList(Items, 'Item', 'Text', Contents);
              Properties.Add('AutoAdd', AutoAdd);
              Properties.Add('Sorted', Sort);
              Properties.Add('AutoComplete', AutoComplete);
              Properties.Add('UseCatalogValues', UseCatalogValues);
            end;
          end;
        end;
      end; // Options/MovieInfo
      with PrepareItem(Items, 'Scripting'), rScripting do
      begin
        Properties.Add('Autorun', ScriptAutorun);
        with PrepareItem(Items, 'Proxy') do
        begin
          Properties.Add('Use', Proxy);
          Properties.Add('Server', ProxyServer);
          Properties.Add('Port', ProxyPort);
          Properties.Add('Username', ProxyUsername);
          Properties.Add('Password', Encrypt(ProxyPassword));
          Properties.Add('KeepConnection', KeepConnection);
          Properties.Add('HTTP10', HTTP10);
        end;
      end; // Options/Scripting
      with PrepareItem(Items, 'Export'), rExport do
      begin
        Properties.Add('LoadTemplate', LoadTemplate);
        Properties.Add('LineBreak', Linebreak);
        Properties.Add('ForcePicSizeW', ForcePicSizeW);
        Properties.Add('ForcePicSizeH', ForcePicSizeH);
        Properties.Add('SQLDate', SQLDate);
        Properties.Add('RememberLastFile', RememberLastFile);
        Properties.Add('LastFile', LastFile);
        Properties.Add('OpenExportedFile', OpenExportedFile);
        Properties.Add('ExpFileStyle', ExpFileStyle);
        Properties.Add('ExpFileExt', ExpFileExt);
        Properties.Add('AddZeroes', AddZeroes);
        Properties.Add('CopyPictures', CopyPictures);
        Properties.Add('CopyPicturesNew', CopyPicturesNew);
      end; // Options/Export
      with PrepareItem(Items, 'Folders') do
      begin
        Properties.Add('SameForAll', SameFolderForAll);
        Items.Clear;
        for i := 0 to length(rFolders)-1 do
          with rFolders[i] do
            rFolders[i].Save(Items, i);
      end; // Options/Folders
      with PrepareItem(Items, 'Language'), rLanguage do
      begin
        Properties.Add('File', Language);
      end; // Options/Languages
    end; // Options
    with PrepareItem(Items, 'CustomFieldsManager'), rCustomFieldsManager do
    begin
      Properties.Add('WindowWidth', WindowWidth);
      Properties.Add('WindowHeight', WindowHeight);
    end; // CustomFieldsManager
  end;
  try
    SetCurrentDir(strDirApp);
    FXml.SaveToFile(strFileSettings);
  except
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

function TSettings.GetLanguageFile: string;
begin
  if rOptions.rLanguage.Language = '' then
    Result := ''
  else
    result := strDirApp + strDirLanguages + rOptions.rLanguage.Language+'.lng'
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

function TSettings.GetFilesSizeUnit(const ByteString: string): string;
begin
  case rOptions.rMovieInformation.ImportSizeUnit of
    fsuGB:   Result := 'G' + ByteString;
    fsuMB:   Result := 'M' + ByteString;
    fsuKB:   Result := 'K' + ByteString;
  else
    Result := ByteString;
  end;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

end.
